#ifndef LLF_CMLA_H
#define LLF_CMLA_H

/*
* All the includes that are needed for code using this module to
* compile correctly should be #included here.
*/

#include "CE2_public.h"
#include "CE2_CMLA.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*
*  Object name     :  LLF_CMLA.h
*  State           :  %state%
*  Creation date   :  29.05.2007
*  Last modified   :  %modify_time%
*/
/** @file
* \brief contains declarations of low level functions for CMLA.
*
*  \version LLF_CMLA.h#1:cinc:1
* \author Yermalayeu Ihar
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd. All rights reserved.
*/

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Structs *****************************/
/************************ Public Variables ********************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  LLF_CMLA_KDF
*
*  @param[in]  X        - Seed value, an octet string of length 128.
*  @param[out] KEK      - Key Encryption Key, an octet string of length 16.
*  
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  CMLA Key Derivation Function produce the Key Encrypting Key [CMLA v1.0-051221].
*  LLF_CMLA_KDF operates on an octet string of length 128.
*  The input splits in two blocks of equal length. A constant value is concatenated
*  to the first block. Then SHA-1 is applied to the preceding result. The result
*  and two integers derived from the second block are used in a modulus calculation
*  to produce the Key Encrypting Key. The output shall be 16 bytes.
*
*  \b 
* Algorithm:
*  -# Let X = X0 || X1, where each Xi consists of 64 bytes octet for i = 0, 1.
*  -# Let C=(0x)00 00 00 01 (4 bytes).
*  -# Compute Y = SHA-1(X0||C) (160 bits).
*  -# Let A be the first 32 bytes and B be the last 32 bytes of X1, respectively.
*  -# Regarding Y, A, and B as integers, derive key encryption key KEK taking the 
*     least significant 128 bits of A*Y + B mod p, where p = 2^192 - 2^64 - 1.
***************************************************************/
CE2Error_t LLF_CMLA_KDF(CE2_CMLA_SEED_t      X ,       /*in*/
                        CE2_CMLA_KEK_t       KEK      /*out*/);

/**
****************************************************************
* Function Name: 
*  LLF_CMLA_Wrap
*
*  @param[in]  KEK      - Key Encryption Key, an octet string of length 16 bytes.
*  @param[in]  Key      - Plain key, an octet string of length 32 bytes.
*  @param[out] WrapKey  - Wrapped Key, an octet string of length 40 bytes.
*  
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  The LLF_CMLA_Wrap primitive composing CE2 AES Key Wrap Algorithm
*  and CMLA_DDT_Perm according to CMLA algorithm [CMLA v1.0-051221].
*
*  \b 
* Algorithm:
*  -# Let Key = K0 || K1, where each Ki consists of 16 bytes octet for i = 0, 1. 
*  -# Apply LLF_CMLA_DDT_Perm to the first 8 bytes of Ki, keeping the rest of 
*     Ki unchanged, to produce 16-byte octet string ki for i = 0, 1. 
*  -# Let k = k0 || k1. 
*  -# Compute CE2_AES_Wrap(KEK, k) to produce an octet string WrapKey of 40 bytes. 
***************************************************************/
CE2Error_t LLF_CMLA_Wrap(CE2_CMLA_KEK_t            KEK,    /*in*/
                         CE2_CMLA_UNWRAPPED_KEY_t  Key,    /*in*/
                         CE2_CMLA_WRAPPED_KEY_t    WrapKey /*out*/ );

/**
****************************************************************
* Function Name: 
*  LLF_CMLA_Unwrap
*
*  @param[in]  KEK      - Key Encryption Key, an octet string of length 16 bytes.
*  @param[in]  WrapKey  - Wrapped Key, an octet string of length 40 bytes.
*  @param[out] Key      - Plain key, an octet string of length 32 bytes.
*  
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  The LLF_CMLA_Unwrap primitive performs inverse CE2_CMLA_Wrap 
*  transformation [CMLA v1.0-051221].
*
*  \b 
* Algorithm:
*  -# Compute CE2_AES_Unwrap(KEK, WrapKey) to produce an octet string k of 
*     length 32 octets.
*  -# Let k = k0||k1, where each ki consists of 16 bytes for i = 0, 1.
*  -# Apply LLF_CMLA_DDT_Perm_Inv to the first 8 bytes of ki, keeping the 
*     rest of ki unchanged, to produce 16-byte octet string Ki for i = 0, 1.
*  -# Let Key = K0 || K1.
***************************************************************/
CE2Error_t LLF_CMLA_Unwrap(CE2_CMLA_KEK_t            KEK,      /*in*/
                           CE2_CMLA_WRAPPED_KEY_t    WrapKey,  /*in*/
                           CE2_CMLA_UNWRAPPED_KEY_t  Key       /*out*/ );

/**
****************************************************************
* Function Name: 
*  LLF_CMLA_RSA_Encrypt
*
*  @param[in]  PublKey     - A pointer to to structure containing user RSA Public Key.
*  @param[in]  Message	   - A pointer to message of length 128 bytes.
*  @param[out] EncrMessage - A pointer to output encrypted message of length 128 bytes.
*  
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  The LLF_CMLA_RSA_Encrypt primitive implements algorithm CMLA_RSA_Encrypt
*  [CMLA v1.0-051221].
*
*  \b 
* Algorithm:
*  -# 
***************************************************************/
CE2Error_t  LLF_CMLA_RSA_Encrypt(CE2_RSAUserPubKey_t      *UserPublKey_ptr, /*in*/
                                 CE2_CMLA_RSA_MESSAGE_t    Message,         /*in*/
                                 CE2_CMLA_RSA_MESSAGE_t    EncrMessage     /*out*/);

/**
****************************************************************
* Function Name: 
*  LLF_CMLA_RSA_Decrypt
*
*  @param[in]  PrivKey     - A pointer to to structure containing user RSA Private Key.
*  @param[in]  EncrMessage - A pointer to input encrypted message of length 128 bytes.
*  @param[out] Message     - A pointer to output decrypted message of length 128 bytes.
*  
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  The LLF_CMLA_RSA_Decrypt primitive implements algorithm CMLA_RSA_Decrypt
*  [CMLA v1.0-051221].
*
*  \b 
* Algorithm:
*  -#  
***************************************************************/
CE2Error_t  LLF_CMLA_RSA_Decrypt(CE2_RSAUserPrivKey_t     *UserPrivKey_ptr, /*in*/
                                 CE2_CMLA_RSA_MESSAGE_t    EncrMessage,     /*in*/
                                 CE2_CMLA_RSA_MESSAGE_t    Message         /*out*/);

#ifdef __cplusplus
}
#endif

#endif /* LLF_CMLA_H */
